<?php

use Drupal\node\Entity\NodeType;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\field\Entity\FieldConfig;
use Drupal\block_content\Entity\BlockContentType;
use Drupal\system\Entity\Menu;
use Drupal\menu_link_content\Entity\MenuLinkContent;
use Drupal\views\Entity\View;

if (PHP_SAPI !== 'cli') {
  throw new \Exception('This script must be run via Drush: drush scr custom_site_setup.php');
}

print "=== Custom Site Setup Script (News / Announcements / Events / Org Info / Blocks / Views) ===\n";

/**
 * Helper: Create node type if not exists.
 */
function create_node_type($type, $label, $description = '') {
  if (NodeType::load($type)) {
    print "Node type '$type' already exists, skipping.\n";
    return;
  }
  $node_type = NodeType::create([
    'type' => $type,
    'name' => $label,
    'description' => $description,
  ]);
  $node_type->save();
  print "Created node type: $type\n";
}

/**
 * Helper: Create block content type if not exists.
 */
function create_block_type($id, $label, $description = '') {
  if (BlockContentType::load($id)) {
    print "Block type '$id' already exists, skipping.\n";
    return;
  }
  $block_type = BlockContentType::create([
    'id' => $id,
    'label' => $label,
    'description' => $description,
  ]);
  $block_type->save();
  print "Created block content type: $id\n";
}

/**
 * Helper: Create Field Storage.
 */
function create_field_storage($entity_type, $field_name, $field_type, array $settings = [], $cardinality = 1) {
  if (FieldStorageConfig::loadByName($entity_type, $field_name)) {
    print "Field storage '$field_name' on '$entity_type' already exists, skipping.\n";
    return;
  }
  $storage = FieldStorageConfig::create([
    'field_name'   => $field_name,
    'entity_type'  => $entity_type,
    'type'         => $field_type,
    'settings'     => $settings,
    'cardinality'  => $cardinality,
  ]);
  $storage->save();
  print "Created field storage: $entity_type.$field_name\n";
}

/**
 * Helper: Create Field Instance.
 */
function create_field_instance($entity_type, $bundle, $field_name, $label, array $settings = [], array $widget_settings = []) {
  if (FieldConfig::loadByName($entity_type, $bundle, $field_name)) {
    print "Field config '$field_name' on bundle '$bundle' already exists, skipping.\n";
    return;
  }

  $config = FieldConfig::create([
    'field_name' => $field_name,
    'entity_type' => $entity_type,
    'bundle' => $bundle,
    'label' => $label,
    'settings' => $settings,
  ]);
  $config->save();
  print "Created field instance: $entity_type.$bundle.$field_name\n";
}

/**
 * Helper: Create menu link.
 */
function create_menu_link($menu_name, $title, $route_name = NULL, $url = NULL, $weight = 0) {
  // If url is given, create a custom path link.
  $existing = \Drupal::entityTypeManager()
    ->getStorage('menu_link_content')
    ->loadByProperties(['title' => $title, 'menu_name' => $menu_name]);
  if ($existing) {
    print "Menu link '$title' in menu '$menu_name' already exists, skipping.\n";
    return;
  }

  if ($url) {
    $link = MenuLinkContent::create([
      'title' => $title,
      'link' => ['uri' => 'internal:' . $url],
      'menu_name' => $menu_name,
      'weight' => $weight,
    ]);
  }
  elseif ($route_name) {
    $link = MenuLinkContent::create([
      'title' => $title,
      'link' => ['uri' => 'route:' . $route_name],
      'menu_name' => $menu_name,
      'weight' => $weight,
    ]);
  }
  else {
    print "No route or URL for menu link '$title', skipping.\n";
    return;
  }

  $link->save();
  print "Created menu link '$title' in menu '$menu_name'.\n";
}

/**
 * Helper: Create a basic view (page + optional block) for one content type.
 */
function create_basic_view($id, $label, $content_type, $path, $page_title, $sort_field, $has_block = TRUE) {
  if (View::load($id)) {
    print "View '$id' already exists, skipping.\n";
    return;
  }

  $view = View::create([
    'id' => $id,
    'label' => $label,
    'base_table' => 'node_field_data',
    'core' => '8.x',
    'display' => [],
  ]);

  // Page display.
  $page_display = [
    'display_plugin' => 'page',
    'id' => 'page_1',
    'display_title' => 'Page',
    'position' => 0,
    'display_options' => [
      'path' => $path,
      'title' => $page_title,
      'access' => [
        'type' => 'perm',
        'options' => ['perm' => 'access content'],
      ],
      'query' => ['type' => 'views_query'],
      'pager' => [
        'type' => 'full',
        'options' => ['items_per_page' => 10],
      ],
      'filters' => [
        'status' => [
          'id' => 'status',
          'table' => 'node_field_data',
          'field' => 'status',
          'value' => 1,
          'plugin_id' => 'boolean',
        ],
        'type' => [
          'id' => 'type',
          'table' => 'node_field_data',
          'field' => 'type',
          'value' => [$content_type => $content_type],
          'plugin_id' => 'bundle',
        ],
      ],
      'sorts' => [
        $sort_field => [
          'id' => $sort_field,
          'table' => "node__{$sort_field}",
          'field' => $sort_field,
          'order' => 'DESC',
          'plugin_id' => 'standard',
        ],
      ],
      'row' => [
        'type' => 'fields',
      ],
      'fields' => [
        'title' => [
          'id' => 'title',
          'table' => 'node_field_data',
          'field' => 'title',
          'label' => '',
          'plugin_id' => 'field',
        ],
      ],
    ],
  ];

  $view->set('display', ['default' => [
    'id' => 'default',
    'display_title' => 'Master',
    'display_plugin' => 'default',
    'position' => 0,
    'display_options' => [
      'access' => [
        'type' => 'perm',
        'options' => ['perm' => 'access content'],
      ],
      'query' => ['type' => 'views_query'],
      'pager' => [
        'type' => 'full',
        'options' => ['items_per_page' => 10],
      ],
      'filters' => [
        'status' => [
          'id' => 'status',
          'table' => 'node_field_data',
          'field' => 'status',
          'value' => 1,
          'plugin_id' => 'boolean',
        ],
        'type' => [
          'id' => 'type',
          'table' => 'node_field_data',
          'field' => 'type',
          'value' => [$content_type => $content_type],
          'plugin_id' => 'bundle',
        ],
      ],
      'row' => [
        'type' => 'fields',
      ],
      'fields' => [
        'title' => [
          'id' => 'title',
          'table' => 'node_field_data',
          'field' => 'title',
          'label' => '',
          'plugin_id' => 'field',
        ],
      ],
    ],
  ]]);

  $view->setDisplay('page_1', $page_display);

  if ($has_block) {
    $block_display = [
      'display_plugin' => 'block',
      'id' => 'block_1',
      'display_title' => 'Block',
      'position' => 1,
      'display_options' => [
        'pager' => [
          'type' => 'some',
          'options' => ['items_per_page' => 3],
        ],
      ],
    ];
    $view->setDisplay('block_1', $block_display);
  }

  $view->save();
  print "Created view: $id\n";
}

/* ==========================================================================
 * 1) CONTENT TYPES
 * ======================================================================== */

print "\n--- Creating Content Types ---\n";

create_node_type('news', 'News', 'Site news items');
create_node_type('announcement', 'Announcement', 'Official site announcements');
create_node_type('event', 'Event', 'Events and activities');
create_node_type('org_info_section', 'Organization Info Section', 'Vision, mission, goals, structure');
create_node_type('team_activity', 'Team Activity', 'Team activities and contributions');


/* ==========================================================================
 * 2) BLOCK CONTENT TYPES
 * ======================================================================== */

print "\n--- Creating Block Content Types ---\n";

create_block_type('front_hero_video', 'Front Hero Video', 'Homepage hero video section');
create_block_type('front_stats', 'Front Statistics', 'Homepage statistics (members, activities, partnerships)');
create_block_type('regions_showcase', 'Regions Showcase', 'Homepage regions showcase (Central, Eastern, Western, Southern)');
create_block_type('social_links', 'Social Links', 'Footer social media links');


/* ==========================================================================
 * 3) FIELDS FOR CONTENT TYPES
 * ======================================================================== */

print "\n--- Creating Fields: Content Types ---\n";

/* === NEWS === */
create_field_storage('node', 'field_news_date', 'datetime', ['datetime_type' => 'date']);
create_field_instance('node', 'news', 'field_news_date', 'News date');

create_field_storage('node', 'field_news_image', 'image');
create_field_instance('node', 'news', 'field_news_image', 'News image');

create_field_storage('node', 'field_news_region', 'entity_reference', [
  'target_type' => 'taxonomy_term',
  'handler' => 'default',
]);
create_field_instance('node', 'news', 'field_news_region', 'Region');

/* === ANNOUNCEMENT === */
create_field_storage('node', 'field_announcement_start_date', 'datetime', ['datetime_type' => 'date']);
create_field_instance('node', 'announcement', 'field_announcement_start_date', 'Start date');

create_field_storage('node', 'field_announcement_end_date', 'datetime', ['datetime_type' => 'date']);
create_field_instance('node', 'announcement', 'field_announcement_end_date', 'End date');

create_field_storage('node', 'field_announcement_attachment', 'file');
create_field_instance('node', 'announcement', 'field_announcement_attachment', 'Attachment');

/* === EVENT === */
create_field_storage('node', 'field_event_start', 'datetime', ['datetime_type' => 'datetime']);
create_field_instance('node', 'event', 'field_event_start', 'Event start');

create_field_storage('node', 'field_event_end', 'datetime', ['datetime_type' => 'datetime']);
create_field_instance('node', 'event', 'field_event_end', 'Event end');

create_field_storage('node', 'field_event_location', 'string');
create_field_instance('node', 'event', 'field_event_location', 'Location');

create_field_storage('node', 'field_event_registration_link', 'link');
create_field_instance('node', 'event', 'field_event_registration_link', 'Registration link');

/* === ORG INFO SECTION === */
create_field_storage('node', 'field_org_section_type', 'list_string', [
  'allowed_values' => [
    'vision'    => 'Vision',
    'mission'   => 'Mission',
    'goals'     => 'Goals',
    'structure' => 'Structure',
  ],
]);
create_field_instance('node', 'org_info_section', 'field_org_section_type', 'Section type');

create_field_storage('node', 'field_org_image', 'image');
create_field_instance('node', 'org_info_section', 'field_org_image', 'Section image');

create_field_storage('node', 'field_org_weight', 'integer');
create_field_instance('node', 'org_info_section', 'field_org_weight', 'Weight');

/* === TEAM ACTIVITY === */
create_field_storage('node', 'field_activity_date', 'datetime', ['datetime_type' => 'date']);
create_field_instance('node', 'team_activity', 'field_activity_date', 'Activity date');

create_field_storage('node', 'field_activity_media', 'entity_reference', [
  'target_type' => 'media',
  'handler' => 'default:media',
], -1);
create_field_instance('node', 'team_activity', 'field_activity_media', 'Activity media');

create_field_storage('node', 'field_activity_external_link', 'link');
create_field_instance('node', 'team_activity', 'field_activity_external_link', 'External link');


/* ==========================================================================
 * 4) FIELDS FOR BLOCK TYPES
 * ======================================================================== */

print "\n--- Creating Fields: Block Types ---\n";

/* === FRONT HERO VIDEO BLOCK === */
create_field_storage('block_content', 'field_hero_title', 'string');
create_field_instance('block_content', 'front_hero_video', 'field_hero_title', 'Hero title');

create_field_storage('block_content', 'field_hero_subtitle', 'text_long');
create_field_instance('block_content', 'front_hero_video', 'field_hero_subtitle', 'Hero subtitle');

create_field_storage('block_content', 'field_hero_media', 'entity_reference', [
  'target_type' => 'media',
  'handler' => 'default:media',
]);
create_field_instance('block_content', 'front_hero_video', 'field_hero_media', 'Hero media');

create_field_storage('block_content', 'field_hero_fallback_image', 'image');
create_field_instance('block_content', 'front_hero_video', 'field_hero_fallback_image', 'Fallback image');

/* === FRONT STATS BLOCK === */
create_field_storage('block_content', 'field_stats_members', 'integer');
create_field_instance('block_content', 'front_stats', 'field_stats_members', 'Members count');

create_field_storage('block_content', 'field_stats_activities', 'integer');
create_field_instance('block_content', 'front_stats', 'field_stats_activities', 'Activities count');

create_field_storage('block_content', 'field_stats_partnerships', 'integer');
create_field_instance('block_content', 'front_stats', 'field_stats_partnerships', 'Partnerships count');

/* === REGIONS SHOWCASE BLOCK === */
create_field_storage('block_content', 'field_region_titles', 'string', [], -1);
create_field_instance('block_content', 'regions_showcase', 'field_region_titles', 'Region titles');

create_field_storage('block_content', 'field_region_descriptions', 'text_long', [], -1);
create_field_instance('block_content', 'regions_showcase', 'field_region_descriptions', 'Region descriptions');

create_field_storage('block_content', 'field_region_links', 'link', [], -1);
create_field_instance('block_content', 'regions_showcase', 'field_region_links', 'Region links');

/* === SOCIAL LINKS BLOCK === */
create_field_storage('block_content', 'field_social_platform', 'list_string', [
  'allowed_values' => [
    'tiktok'   => 'TikTok',
    'instagram'=> 'Instagram',
    'snapchat' => 'Snapchat',
    'youtube'  => 'YouTube',
    'twitter'  => 'X / Twitter',
  ],
], -1);
create_field_instance('block_content', 'social_links', 'field_social_platform', 'Platform name');

create_field_storage('block_content', 'field_social_url', 'link', [], -1);
create_field_instance('block_content', 'social_links', 'field_social_url', 'Platform URL');

create_field_storage('block_content', 'field_social_icon', 'string', [], -1);
create_field_instance('block_content', 'social_links', 'field_social_icon', 'Platform icon');


/* ==========================================================================
 * 5) MENUS & LINKS
 * ======================================================================== */

print "\n--- Creating Menu Links ---\n";

// Main menu usually exists by default in Drupal, we just add links.
create_menu_link('main', 'Home', NULL, '/', -10);
create_menu_link('main', 'About us', NULL, '/about', -5);
create_menu_link('main', 'Team activities', NULL, '/activities', 0);
create_menu_link('main', 'Media center', NULL, '/media-center', 5);

// Footer menu.
if (!Menu::load('footer')) {
  $footer_menu = Menu::create([
    'id' => 'footer',
    'label' => 'Footer navigation',
    'description' => 'Footer links',
  ]);
  $footer_menu->save();
  print "Created footer menu.\n";
}

create_menu_link('footer', 'Contact us', NULL, '/contact', 0);


/* ==========================================================================
 * 6) VIEWS (BASIC)
 * ======================================================================== */

print "\n--- Creating Views (basic structures) ---\n";

// News view (page + block)
create_basic_view(
  'view_news',
  'News',
  'news',
  '/news',
  'News',
  'field_news_date'
);

// Announcements view (page + block)
create_basic_view(
  'view_announcements',
  'Announcements',
  'announcement',
  '/announcements',
  'Announcements',
  'field_announcement_start_date'
);

// Events view (page + block)
create_basic_view(
  'view_events',
  'Events',
  'event',
  '/events',
  'Events',
  'field_event_start'
);

// Org info sections view (About + homepage block)
if (!View::load('view_org_info_sections')) {
  $view = View::create([
    'id' => 'view_org_info_sections',
    'label' => 'Organization info sections',
    'base_table' => 'node_field_data',
    'core' => '8.x',
    'display' => [],
  ]);

  $view->set('display', [
    'default' => [
      'id' => 'default',
      'display_plugin' => 'default',
      'display_title' => 'Master',
      'position' => 0,
      'display_options' => [
        'access' => [
          'type' => 'perm',
          'options' => ['perm' => 'access content'],
        ],
        'filters' => [
          'status' => [
            'id' => 'status',
            'table' => 'node_field_data',
            'field' => 'status',
            'value' => 1,
            'plugin_id' => 'boolean',
          ],
          'type' => [
            'id' => 'type',
            'table' => 'node_field_data',
            'field' => 'type',
            'value' => ['org_info_section' => 'org_info_section'],
            'plugin_id' => 'bundle',
          ],
        ],
        'sorts' => [
          'field_org_weight' => [
            'id' => 'field_org_weight',
            'table' => 'node__field_org_weight',
            'field' => 'field_org_weight',
            'order' => 'ASC',
            'plugin_id' => 'standard',
          ],
        ],
        'row' => ['type' => 'fields'],
        'fields' => [
          'title' => [
            'id' => 'title',
            'table' => 'node_field_data',
            'field' => 'title',
            'label' => '',
            'plugin_id' => 'field',
          ],
          'body' => [
            'id' => 'body',
            'table' => 'node__body',
            'field' => 'body',
            'label' => '',
            'plugin_id' => 'field',
          ],
        ],
      ],
    ],
    'page_1' => [
      'id' => 'page_1',
      'display_plugin' => 'page',
      'display_title' => 'Page',
      'position' => 1,
      'display_options' => [
        'path' => '/about/sections',
        'title' => 'Organization info sections',
      ],
    ],
    'block_1' => [
      'id' => 'block_1',
      'display_plugin' => 'block',
      'display_title' => 'Block',
      'position' => 2,
      'display_options' => [],
    ],
  ]);

  $view->save();
  print "Created view: view_org_info_sections\n";
}

// Team activities view (optional).
create_basic_view(
  'view_team_activities',
  'Team activities',
  'team_activity',
  '/activities',
  'Team activities',
  'field_activity_date'
);

print "\n=== DONE. Enable layout / manage display via UI as needed. ===\n";

